﻿// ============================================================================
// Movo Network API Sample
//
// <CS_ApiSample3.cs>
//
// Copyright (c) 2009 Servoland corporation.
// All rights reserved.
//
// 目的：
//		ドライバのロギングデータをPCに高速転送します。
//
// Version情報：					  -- 追加, 修正項目 --
//		Version 1.0		2009.08.10	: First construction.
//
// ============================================================================
using System;
using System.IO;
using MovoEtherVB.Servoland;


namespace CS_ApiSample3
{
    class CS_ApiSample3
    {
        private static MovoAPI api;
        private static int iDevSel;
        private static Char ID;

        // =====================================================================
        // エントリーポイント
        // =====================================================================
        static void Main(string[] args)
        {
            int iRc;
            String line;

            iRc = Initialize();
            if (iRc != 0)
            {
                // 初期化失敗
                Console.WriteLine("Initialization failed (Hit Enter to exit)");
                line = Console.ReadLine();
                return;
            }

            Logging();

        }

        // =====================================================================
        // 初期化
        // =====================================================================
        static int Initialize()
        {
            String strBuf = ""; // 文字列バッファ
            int i;              // Counter
            int iRc;            // Return code
            int IFNum;          // Ethernet I/Fの数
            int IFSel;          // 使用するEthetnet I/F
            String strIP = "";  // IPアドレス(文字列)
            int pIP=0;          // IPアドレス(整数)
            String strSN = "";  // サブネットマスク(文字列)
            int pSN=0;          // サブネットマスク(整数)

            api = new MovoAPI(); // MovoAPIクラスのインスタンス生成

            iDevSel = 0; // 通信デバイス=Ethernet

            // APIのバージョン取得
            api.GetLibVersion(ref strBuf);
            strBuf = "Network API version " + strBuf + "\n";

            Console.WriteLine(strBuf);

            // I/Fを設定(Ethernet使用時)
            if (iDevSel == 0)
            {
                // 使用可能なEthernet I/Fを検索
                IFNum = api.SearchInterface(iDevSel);
                if (IFNum < 0)
                {
                    Console.WriteLine("No ethernet I/F found");
                    return -1;
                }

                Console.WriteLine(IFNum.ToString() + " ethernet I/F found");

                // 各I/Fの情報を表示
                for (i = 0; i < IFNum; i++)
                {
                    api.GetHostIP(iDevSel, i, ref strIP, ref pIP);
                    api.GetHostSN(iDevSel, i, ref strSN, ref pSN);
                    Console.WriteLine(i.ToString() + " : " + String.Format("{0,-16}", strIP) + "(" + String.Format("{0,-16}", strSN) + ")");
                }

                if (IFNum == 1)
                {
                    // I/Fが１つのときは自動選択
                    IFSel = 0;
                }
                else
                {
                    // I/Fが２つ以上のときはユーザが選択
                    do
                    {
                        Console.Write("Select I/F to use [0-" + (IFNum - 1).ToString() + "]> ");
                        strBuf = Console.ReadLine();
                        try
                        {
                            IFSel = Convert.ToInt32(strBuf);
                        }
                        catch
                        {
                            IFSel = -1;
                        }
                    } while (IFSel < 0 || IFSel >= IFNum);

                }
                Console.WriteLine("I/F " + IFSel.ToString() + " selected\n");

                // Ethernet I/Fを設定
                api.SetInterface(iDevSel, IFSel);
            }

            // 接続されたドライバの検索
            iRc = api.SearchDevice(iDevSel, ref strBuf);
            if (iRc <= 0)
            {
                Console.WriteLine("No driver found");
                return -1;
            }
            ID = strBuf.ToCharArray()[0];

            // 接続可能なすべてのドライバに接続
            iRc = api.ConnectAll(iDevSel, ref strBuf);
            if (iRc > 0)
            {
                Console.WriteLine("Driver " + strBuf + " connected");
            }
            else
            {
                Console.WriteLine("No driver found");
                return -1;
            }
            return 0;
        }


        // =====================================================================
        // 高速転送によるロギングデータの取得
        // =====================================================================
        static void Logging()
        {
            int i;
            int iRc;
            int[] data=new int[4000]; 	// ロギングデータ
            String strBuf;              // 文字列バッファ
            MOVO_VAR2 m_var;	        // 高速転送のための構造体
            COM_BIN cb_logp;	        // コマンドアドレス構造体
            int logp;               	// ロギングポインタ

            // ロギング領域の先頭アドレスを設定
            m_var.address = 0x0008C000;

            // 軸名設定
            api.SetComID(iDevSel, ID);

            // LOGPにコマンドを使わずにアクセスするためアドレスを取得
            strBuf = "LOGP";
            Console.WriteLine(strBuf);
            cb_logp.com_addr = 0;
            cb_logp.com_data = 0;
            cb_logp.com_type = 0;
            iRc = api.GetComAddr(iDevSel, ref strBuf, ref cb_logp);
            if (iRc == MovoAPI.MOVO_FAILURE)
            {
                Console.WriteLine("Error occured at LOGP");
                return;
            }

            // ロギング領域初期化 ("LC")
            strBuf = "LC";
            Console.WriteLine(strBuf);
            iRc = api.SendCom(iDevSel, ref strBuf);
            if (iRc == MovoAPI.MOVO_FAILURE)
            {
                Console.WriteLine("Error occured at LC");
                return;
            }

            // LC完了待ち(LC完了時LOGPはロギング領域の先頭を指す)
            do
            {
                System.Threading.Thread.Sleep(100);	// insert interval
                // コマンドを使わずLOGPを取得(コマンド使用時より高速)
                logp = 0;
                iRc = api.Movo_ComAddrIO_32(iDevSel, ID, 1, ref cb_logp, ref logp);
                if (iRc == MovoAPI.MOVO_FAILURE)
                {
                    Console.WriteLine("Error occured at LOGP through Movo_ComAddrIO");
                    return;
                }
                strBuf = "0x" + (Convert.ToString(logp, 16)).PadLeft(8, '0');                
                Console.WriteLine(strBuf.ToUpper());
            } while (m_var.address != logp);
            Console.WriteLine("LC is over\n");

            // ロギング周期を1msに設定("LTIME=1")
            strBuf = "LTIME=1";
            Console.WriteLine(strBuf);
            iRc = api.SendCom(iDevSel, ref strBuf);
            if (iRc == MovoAPI.MOVO_FAILURE)
            {
                Console.WriteLine("Error occured at LTIME");
                return;
            }

            // ロギング開始 ("L")
            strBuf = "L";
            Console.WriteLine(strBuf);
            iRc = api.SendCom(iDevSel, ref strBuf);
            if (iRc == MovoAPI.MOVO_FAILURE)
            {
                Console.WriteLine("Error occured at L");
                return;
            }

            // ロギング終了判定
            do
            {
                System.Threading.Thread.Sleep(100);	// insert interval
                // コマンドを使わずLOGPを取得(コマンド使用時より高速)
                logp = 0;
                iRc = api.Movo_ComAddrIO_32(iDevSel, ID, 1, ref cb_logp, ref logp);
                if (iRc == MovoAPI.MOVO_FAILURE)
                {
                    Console.WriteLine("Error occured at LOGP through Movo_ComAddrIO");
                    return;
                }
                strBuf = "0x" + (Convert.ToString(logp, 16)).PadLeft(8, '0');
                Console.WriteLine(strBuf.ToUpper());
            } while (m_var.address != logp);
            Console.WriteLine("L is over\n");

            // ------ ここから高速転送 ------

            /*
             * 注意：
             *  高速転送はコマンドを使用しないため、その解析時間が無くなります。
             *  これによりコマンド経由での転送より高速になります。
             *  "L"コマンドがドライバに送信され、ロギングが進行中の場合、高速転送は
             *  ドライバのロギングを追い越してしまう恐れがあります。
             *  これを避けるためにはLOGPを監視しながらデータ転送を行う必要があります。
             *  このサンプルプログラムではロギング終了後にデータを転送するので、
             *  追い越しは発生しません。
             */

            // MOVO_VAR2構造体の設定
            m_var.target = 0;						// 通常変数
            m_var.purpose = MovoAPI.MOVO_GET32;		// 32ビット変数としてロギングデータを取得
            m_var.total = 4000;					    // 4ch x 1000行
            m_var.offset = 0;						// 開始時のオフセットなし
            m_var.xfersize = 0;				    	// 1転送あたりのサイズは最大転送量

            // すべてのロギングデータを取得
            while (m_var.total != m_var.offset)
            {
                iRc = api.Movo_DataIO_32(iDevSel, ID, ref m_var, ref data[m_var.offset]);
                if (iRc == MovoAPI.MOVO_FAILURE)
                {
                    Console.WriteLine("Error occured at Movo_DataIO");
                    return;
                }
            }

            // ------ ここまで高速転送 ------

            Console.WriteLine("Save logging data to [log.csv]");

            // ファイル書き込み用のストリーム
            StreamWriter sw = new StreamWriter("log.csv");

            // ロギング変数名取得
            for (i = 1; i <= 4; i++)
            {
                // LOGxN
                strBuf = "LOG" + i.ToString() + "N";
                iRc = api.SendCom(iDevSel, ref strBuf);
                if (iRc == MovoAPI.MOVO_FAILURE)
                {
                    Console.WriteLine("Error occured at LOGxN");
                    return;
                }
                api.RecvCom(iDevSel, ref strBuf);
                sw.Write(strBuf);
                if (i < 4)
                    sw.Write(",");
                else
                    sw.Write("\n");
            }

            // データの書き出し
            for (i = 0; i < 4000; i++)
            {
                sw.Write(data[i].ToString());
                if (i % 4 < 3)
                    sw.Write(",");
                else
                    sw.Write("\n");
            }

            // ストリームを閉じる
            sw.Close();

            Console.WriteLine("Done");
        }
    }
}

