﻿' ============================================================================
' Movo Network API Sample
'
' <VB_ApiSample3.vb>
'
' Copyright (c) 2009 Servoland corporation.
' All rights reserved.
'
' 目的：
'		ドライバのロギングデータをPCに高速転送します。
'
' Version情報：					    -- 追加, 修正項目 --
'		Version 1.0		2009.08.11	: First construction.
'
' ============================================================================
Imports MovoEtherVB.Servoland
Imports System.IO

Module VB_ApiSample3
    Dim iDevSel As Integer
    Dim api As MovoAPI
    Dim ID As Char

    ' =========================================================================
    ' エントリーポイント
    ' =========================================================================
    Sub Main()
        Dim iRc As Integer
        Dim line As String

        iRc = Initialize()
        If iRc <> 0 Then
            ' 初期化失敗
            Console.WriteLine("Initialization failed (Hit Enter to exit)")
            line = Console.ReadLine()
            Return
        End If

        Logging()

    End Sub

    ' =========================================================================
    ' 初期化
    ' =========================================================================
    Function Initialize() As Integer
        Dim Str As String = ""  ' 文字列バッファ
        Dim i As Integer        ' Counter
        Dim iRc As Integer      ' Return code
        Dim IFNum As Integer    ' Ethernet I/Fの数
        Dim IFSel As Integer    ' 使用するEthetnet I/F
        Dim strIP As String = ""    ' IPアドレス(文字列)
        Dim pIP As Integer          ' IPアドレス(整数)
        Dim strSN As String = ""    ' サブネットマスク(文字列)
        Dim pSN As Integer          ' サブネットマスク(整数)

        api = New MovoAPI ' MovoAPIクラスのインスタンス生成

        iDevSel = 0 ' 通信デバイス=Ethernet

        ' APIのバージョン取得
        api.GetLibVersion(Str)
        Str = "Network API version " + Str + ControlChars.NewLine

        Console.WriteLine(Str)

        ' I/Fを設定(Ethernet使用時)
        If iDevSel = 0 Then
            ' 使用可能なEthernet I/Fを検索
            IFNum = api.SearchInterface(iDevSel)
            If IFNum < 0 Then
                Console.WriteLine("No ethernet I/F found")
                Return -1
            End If
            Console.WriteLine(IFNum.ToString + " ethernet I/F found")

            ' 各I/Fの情報を表示
            For i = 0 To IFNum - 1
                api.GetHostIP(iDevSel, i, strIP, pIP)
                api.GetHostSN(iDevSel, i, strSN, pSN)
                Console.WriteLine(i.ToString + " : " + String.Format("{0,-16}", strIP) + "(" + String.Format("{0,-16}", strSN) + ")")
            Next

            If IFNum = 1 Then
                ' I/Fが１つのときは自動選択
                IFSel = 0
            Else
                ' I/Fが２つ以上のときはユーザが選択
                Do
                    Console.Write("Select I/F to use [0-" + (IFNum - 1).ToString + "]> ")
                    Str = Console.ReadLine()
                    Try
                        IFSel = Convert.ToInt32(Str)
                    Catch
                        IFSel = -1
                    End Try
                Loop While IFSel < 0 Or IFSel >= IFNum

            End If
            Console.WriteLine("I/F " + IFSel.ToString + " selected" + ControlChars.NewLine)

            ' Ethernet I/Fを設定
            api.SetInterface(iDevSel, IFSel)

        End If

        ' 接続されたドライバの検索
        iRc = api.SearchDevice(iDevSel, Str)
        If iRc <= 0 Then
            Console.WriteLine("No driver found")
            Return -1
        End If
        ID = Str.ToCharArray()(0)

        ' 接続可能なすべてのドライバに接続
        iRc = api.ConnectAll(iDevSel, Str)
        If iRc > 0 Then
            Console.WriteLine("Driver " + Str + " connected")
        Else
            Console.WriteLine("No driver found")
            Return -1
        End If

        Return 0

    End Function

    ' =========================================================================
    ' 高速転送によるロギングデータの取得
    ' =========================================================================
    Sub Logging()
        Dim i As Integer
        Dim iRc As Integer
        Dim data(4000) As Integer   ' ロギングデータ
        Dim strBuf As String        ' 文字列バッファ
        Dim m_var As MOVO_VAR2      ' 高速転送のための構造体
        Dim cb_logp As COM_BIN      ' コマンドアドレス構造体
        Dim logp As Integer         ' ロギングポインタ

        ' ロギング領域の先頭アドレスを設定
        m_var.address = &H8C000

        ' 軸名設定
        api.SetComID(iDevSel, ID)

        ' LOGPにコマンドを使わずにアクセスするためアドレスを取得
        strBuf = "LOGP"
        Console.WriteLine(strBuf)
        cb_logp.com_addr = 0
        cb_logp.com_data = 0
        cb_logp.com_type = 0
        iRc = api.GetComAddr(iDevSel, strBuf, cb_logp)
        If iRc = MovoAPI.MOVO_FAILURE Then
            Console.WriteLine("Error occured at LOGP")
            Return
        End If

        ' ロギング領域初期化 ("LC")
        strBuf = "LC"
        Console.WriteLine(strBuf)
        iRc = api.SendCom(iDevSel, strBuf)
        If iRc = MovoAPI.MOVO_FAILURE Then
            Console.WriteLine("Error occured at LC")
            Return
        End If

        ' LC完了待ち(LC完了時LOGPはロギング領域の先頭を指す)
        Do

            System.Threading.Thread.Sleep(100)  ' insert interval
            ' コマンドを使わずLOGPを取得(コマンド使用時より高速)
            logp = 0
            iRc = api.Movo_ComAddrIO_32(iDevSel, ID, 1, cb_logp, logp)
            If iRc = MovoAPI.MOVO_FAILURE Then
                Console.WriteLine("Error occured at LOGP through Movo_ComAddrIO")
                Return
            End If
            strBuf = "0x" + (Convert.ToString(logp, 16)).PadLeft(8, "0")
            Console.WriteLine(strBuf.ToUpper())
        Loop While m_var.address <> logp
        Console.WriteLine("LC is over" + ControlChars.NewLine)

        ' ロギング周期を1msに設定("LTIME=1")
        strBuf = "LTIME=1"
        Console.WriteLine(strBuf)
        iRc = api.SendCom(iDevSel, strBuf)
        If iRc = MovoAPI.MOVO_FAILURE Then
            Console.WriteLine("Error occured at LTIME")
            Return
        End If

        ' ロギング開始 ("L")
        strBuf = "L"
        Console.WriteLine(strBuf)
        iRc = api.SendCom(iDevSel, strBuf)
        If iRc = MovoAPI.MOVO_FAILURE Then
            Console.WriteLine("Error occured at L")
            Return
        End If

        ' ロギング終了判定
        Do

            System.Threading.Thread.Sleep(100)  ' insert interval
            ' コマンドを使わずLOGPを取得(コマンド使用時より高速)
            logp = 0
            iRc = api.Movo_ComAddrIO_32(iDevSel, ID, 1, cb_logp, logp)
            If iRc = MovoAPI.MOVO_FAILURE Then
                Console.WriteLine("Error occured at LOGP through Movo_ComAddrIO")
                Return
            End If
            strBuf = "0x" + (Convert.ToString(logp, 16)).PadLeft(8, "0")
            Console.WriteLine(strBuf.ToUpper())
        Loop While m_var.address <> logp
        Console.WriteLine("L is over" + ControlChars.NewLine)

        ' ------ ここから高速転送 ------

        ' 注意：
        '  高速転送はコマンドを使用しないため、その解析時間が無くなります。
        '  これによりコマンド経由での転送より高速になります。
        '  "L"コマンドがドライバに送信され、ロギングが進行中の場合、高速転送は
        '  ドライバのロギングを追い越してしまう恐れがあります。
        '  これを避けるためにはLOGPを監視しながらデータ転送を行う必要があります。
        '  このサンプルプログラムではロギング終了後にデータを転送するので、
        '  追い越しは発生しません。

        ' MOVO_VAR2構造体の設定
        m_var.target = 0                        ' 通常変数
        m_var.purpose = MovoAPI.MOVO_GET32      ' 32ビット変数としてロギングデータを取得
        m_var.total = 4000                      ' 4ch x 1000行
        m_var.offset = 0                        ' 開始時のオフセットなし
        m_var.xfersize = 0                      ' 1転送あたりのサイズは最大転送量

        ' すべてのロギングデータを取得
        Do While m_var.total <> m_var.offset
            iRc = api.Movo_DataIO_32(iDevSel, ID, m_var, data(m_var.offset))
            If iRc = MovoAPI.MOVO_FAILURE Then
                Console.WriteLine("Error occured at Movo_DataIO")
                Return
            End If
        Loop

        ' ------ ここまで高速転送 ------

        Console.WriteLine("Save logging data to [log.csv]")

        ' ファイル書き込み用のストリーム
        Dim sw As StreamWriter
        sw = New StreamWriter("log.csv")

        ' ロギング変数名取得
        For i = 1 To 4
            ' LOGxN
            strBuf = "LOG" + i.ToString() + "N"
            iRc = api.SendCom(iDevSel, strBuf)
            If iRc = MovoAPI.MOVO_FAILURE Then
                Console.WriteLine("Error occured at LOGxN")
                Return
            End If
            api.RecvCom(iDevSel, strBuf)
            sw.Write(strBuf)
            If i < 4 Then
                sw.Write(",")
            Else
                sw.Write(ControlChars.NewLine)
            End If
        Next

        ' データの書き出し
        For i = 0 To 3999

            sw.Write(data(i).ToString())
            If i Mod 4 < 3 Then
                sw.Write(",")
            Else
                sw.Write(ControlChars.NewLine)
            End If
        Next

        ' ストリームを閉じる
        sw.Close()

        Console.WriteLine("Done")

    End Sub
End Module

